/** @file   menupage.h
 * @brief   Declaraction of MenuPage - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_MENUPAGE_H
#define H_WWW_MENUPAGE_H

#include <vector>
#include "overlayitem.h"


namespace WeWantWar {


/** @class  MenuPage
 * @brief   Baseclass of all the menu-classes.
 * @author  Tomi Lamminsaari
 *
 * Every menu-class should be derived from this class.
 */
class MenuPage
{
public:

  ///
  /// Static members and methods
  /// ==========================
  
  /** Returned from 'update()'-method when there's no need to change the
   * menupage.
   */
  static const int NO_PAGE_CHANGE = 0;
  /** Returned from 'update()'-method when we should change the menupage. */
  static const int PAGE_CHANGE = 1;

  /** Defines the datatype for menupage id-codes. These get returned from
   * 'getPageID()'-method.
   */
  typedef int PageID;
  /** ID of mainmenu */
  static const PageID MAINMENU = 0;
  /** ID of Settings-menu */
  static const PageID SETTINGS = 1;
  /** ID of menupage that is shown between the game levels. */
  static const PageID LEVELCHANGE = 2;
  /** ID of Save Game-page. */
  static const PageID SAVESLOT = 3;
  /** ID of Load Game-page */
  static const PageID LOADGAME = 4;
  /** ID of Notification windows (e.g. Game Loaded ) */
  static const PageID NOTIFICATION = 5;
  /** ID of Game Paused - menu */
  static const PageID GAMEPAUSED = 6;
  /** ID of highscores menu */
  static const PageID HIGHSCORES = 7;
  /** ID of enter highscores menu */
  static const PageID ENTERHIGHSCORES = 8;
  


  /** The action states this menupage can be in.
   */
  enum ActionState {
    /** Menupage is opening. Usually playing some animation where the menuitems
     * come to the screen.
     */
    STATE_FADEIN,
    /** Normal visible state where player can access the menu. */
    STATE_VISIBLE,
    /** This menu is closing itself. */
    STATE_FADEOUT
  };
  
  
  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Constructs new MenuPage
   */
  MenuPage();
  
  /** Destructor.
   */
  virtual ~MenuPage();
  


  ///
  /// Public methods
  /// ==============

  /** The update method is pure virtual function and therefore all the
   * actual MenuPages-classes has to implement it. It should handle the
   * keyboard and navigating inside the page.
   * @return    Either NO_PAGE_CHANGE or PAGE_CHANGE depending on should we
   *            exit from this page and change to another page.
   */
  virtual int update() = 0;
  
  /** Redraws this menupage to the screen. This calls the <code> redraw() </code>
   * method that should draw the contents to the double buffer.
   */
  void redrawPage();
  
  /** Adds new OverlayItem to this menupage.
   * @param     pItem             Pointer to new overlay item
   */
  void addOverlayItem( OverlayItem* pItem );
  


  ///
  /// Public GET-methods
  /// ==================
  
  /** Should return the type of the page.
   * @return    ID of this page.
   */
  virtual PageID getPageID() const = 0;
  
  /** Returns the index of the menu option that currently has selection.
   * @return    Index of the menu item that currently has selection.
   */
  virtual int    getSelection() const;

  /** Returns the requested OverlayItem
   * @param     index             Index of the OverlayItem being returned.
   * @return    Pointer to requested item or null-pointer if there is no
   *            item on such index.
   */
  OverlayItem* getOverlayItem( int index ) const;
  
  /** Returns the state this menupage is in.
   * @return    Current state
   */
  ActionState getState() const;
  

protected:

  /** Redraws the menupage. This method must be implemented in derived
   * classes.
   */
  virtual void redraw() = 0;



  ///
  /// Members
  /// =======
  
  /** The scancode of the key that was previously pressed. */
  static int    s_lastKey;
  /** Index of the menuitem that currently has selection. */
  int           m_selection;
  /** A list of overlayed items */
  std::vector<OverlayItem*> m_overlayedItems;
  /** The state this menupage is in. */
  ActionState   m_state;
  
  
private:

  /** Private methods **/

  MenuPage(const MenuPage& rO);
  MenuPage& operator = (const MenuPage& rO);
};

};  // end of namespace


#endif

/**
 * Version history
 * ===============
 * $Log: menupage.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:03+02  lamminsa
 * Initial revision
 *
 */
 
